#!/usr/bin/python3

# logmgrd.py
#
# Copyright (C) 2024 Tuomo Makinen (tjam@users.sourceforge.net)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

# 
#
"""
Node manager for Nuhe - action capable log file monitor

Copyright (C) 2025 Tuomo Makinen

Usage:
  -c  <file>  Configuration file for Nuhe node manager
  -f  Foreground mode
  -r  Redirect logging to stdout
  -v  Show version of Nuhe node manager
  -h  Prints this message
"""

import os
import string
import sys
import time
import getopt
import signal
import select
import hashlib
import socket
import platform
import base64
import json
from errno import EINTR


from cryptography import utils, x509

import OpenSSL
# SSL serversocket class
import serversocket
# ThreadPool class
import ThreadPool


# Wrapper for select 
def do_select (iwtd, owtd, ewtd, timeout=None):
    while True:
        try:
            if timeout != None:
                rl, wl, el = select.select(iwtd, owtd, ewtd, timeout)
            else:
                rl, wl, el = select.select(iwtd, owtd, ewtd)
        except select.error as e:
            # Interrupted system call
            if e[0] == EINTR:
                continue
            else:
                return (None,)
        else:
            break

    return (rl, wl, el)

def wait_connection():
    # Activate server objects
    Sensorobj.do_socket()
    serv_stat = Sensorobj.activate_server(Slistenaddr, Portsensor)
    if serv_stat[0] == -1:
        fatalerror(serv_stat[1])

    Clientobj.do_socket()
    serv_stat = Clientobj.activate_server(Clistenaddr, Portclient)
    if serv_stat[0] == -1:
        fatalerror(serv_stat[1])

    logwriter("Nuhe node manager started")

    # Time to drop privileges
    if (drop_privs () < 1):
        logwriter("Nuhe node manager running with dropped privileges")

    sfd = Sensorobj.sockobj.fileno()
    cfd = Clientobj.sockobj.fileno()

    # Serve forever
    while True:

        slct = do_select([sfd, cfd], [], [])
        if slct[0] == None:
            continue

        if sfd in slct[0]:
            try:
                conn, addr = Sensorobj.accept_conn()
            except:
                pass
            else:
                Pool_t.runTask(handle_sensor_connection, conn, addr[0])

        elif cfd in slct[0]:
            try:
                conn, addr = Clientobj.accept_conn()
            except:
                pass
            else:
                Pool_t.runTask(handle_client_connection, conn, addr[0])


def drop_privs ():
   # setgid/setuid
   if (Process_gid == None or Process_uid == None):
     return 1

   if Process_gid != None:
      try:
          os.setgid(Process_gid)
      except OSError:
          fatalerror("invalid gid specification, exiting")

   if Process_uid != None:
      try:
          os.setuid(Process_uid)
      except OSError:
          fatalerror("invalid uid specification, exiting")

   return 0
 
def handle_sensor_connection(conn, host):

    # my_sensor holds sensor's name for current thread which
    # is responsible for that sensor
    my_sensor = None
   
    start_t = time.time()
    alive_t = time.time()

    if serversocket.UseSSL:
        conn.shutdown()

    conn.close()

def parse_sensor_message(host, message):

    if message[:Oplen] == SN_CLNTHELLO:
        payload = message[Oplen+1:]

        # Bad message payload
        if payload == '':
	        return SN_CLNTHELLO, 1

        return SN_CLNTHELLO, payload


    elif message[:Oplen] == SN_ALIVE:
        # Sensor is alive
        return SN_ALIVE, 0
    elif message[:Oplen] == SN_ALERTMSG:
        # Sensor sends alert data
        payload = message[Oplen:]
        payload = payload.strip()
        return SN_ALERTMSG, payload
    elif message[:Oplen] == SN_DISCONNECT_S:
        # Sensor wants to disconnect
        return SN_DISCONNECT_S, 0
    else:
        return UNKWNMSG, 0

def do_sensor_tasks(conn, sensor):
    task_l = []

    if sensor != None:
        task_l = check_sensor_tasks(sensor)
        if len(task_l) > 0:
            for i in range(len(task_l)):
                if task_l[i][4] == None:
                    tret = send_tasks_to_sensor(conn, task_l[i], sensor)
                    return tret

    return 0
 
def send_tasks_to_sensor(conn, task, sensor):
    return 0

def modify_tasks (task, sensor, rvalue, status):
    # Iterate through Snsrtask_l
    Pool_t.acquireLock(MSnsrtask_l)
    for x in range(len(Snsrtask_l)):
        if Snsrtask_l[x] == task:
            if Snsrtask_l[x][1] == sensor:
                if status == 1:
                    # This is bad task, mark it
                    Snsrtask_l[x][5] = 1
                else:
                    # Mark this task done
                    Snsrtask_l[x][4] = rvalue
            
    Pool_t.releaseLock(MSnsrtask_l)

def check_sensor_tasks(sensor):
    my_tasks = []

    # Poll Snsrtask_l
    Pool_t.acquireLock(MSnsrtask_l)
    for x in range(len(Snsrtask_l)):
        if Snsrtask_l[x][1] == sensor:
           # Ok, some work to do
           if Snsrtask_l[x][4] == None:
               # Add it to list
               my_tasks.append(Snsrtask_l[x])
    
    Pool_t.releaseLock(MSnsrtask_l)

    return my_tasks

def do_opcode_modify (opcode):
    if (opcode not in [REQRJCT, UNKWNMSG, REQFAIL]):
        opcode = Opcmap.get(opcode)

    if not opcode:
        return REQFAIL
    else:
        return opcode

def register_sensor (sname, version, stime):
    return 0

def undo_register_sensor(sname):
   
    return 0

def set_sensor_status(sname, status, version=None, stime=None):
    return 0

def get_sensor_status(sname, full=False):
    if sname == None:
        return -1

    Pool_t.acquireLock(MSensor_d)
    try:
        sstat = Sensor_d[sname]
    except KeyError:
        sstat = -1
    Pool_t.releaseLock(MSensor_d)

    # Return running status, version and startup time
    if full:
        return sstat

    # Return only running status
    return sstat[0]

def log_alertmsg (alertmsg):
    # Strip delimiter
    if alertmsg[-2:] == "\r\n":
        alertmsg = alertmsg[:-2]

    if Alertlog == 1:
        # Logging alerts to flat file
        i = alertmsg.find( "[")
        if (i != -1):
            alertmsg = alertmsg[:i-1] + "] " + alertmsg[i:]
        alertmsg = "[" + alertmsg

        Alrfp.write(alertmsg + "\n")
        Alrfp.flush()
        
def handle_client_connection(conn, host):

    me = Pool_t.getThreadId()
    client_sensor = None
    authflag = False

    logwriter("*** Client from: " + host + " connected")

    while True:
        # Check if Shutdown flag is set
        if Shutdown == True:
            Clientobj.write_socket(conn, NC_DISCONNECT_N)
            break

        slct = do_select([conn], [], [], 5.0)
        if slct[0] == None:
            break

        if not slct[0]:
            continue

        message = Clientobj.read_socket(conn)
        if not message:
            # Connection is broken
            logwriter("Client: " + host + \
                       " has closed connection prematurely")
            break
         
        else:
            # Password authenticate client
            if Clientauth == 0 and authflag == False:
                if message[:Oplen] != "101":
                    break

                message = message[Oplen:].strip().split( ":", 1)
                if len(message) != 2:
                    break

                rval, username, sensor_ac = passwd_authenticate_client (message[0], message[1])
                if rval == 0:
                    authflag = True
                    Clientobj.write_socket(conn, NC_PASSWDAUTH + " OK")
                    logwriter("User: " + message[0] + " from " + host + " authenticated")
                    continue
                else:
                    logwriter("Authentication failure for user: " + message[0] + " from " + host)
                    Clientobj.write_socket(conn, REQFAIL)
                    break
            
            # Process client message
            msgtype, rval = parse_client_message(username, sensor_ac, host, message, client_sensor)
            payload       = rval

            logwriter("Parsed client message:"+str(msgtype)+" val:"+str(rval))
            if msgtype == REQFAIL:
                Clientobj.write_socket(conn, REQFAIL)
            elif msgtype == REQRJCT:
                Clientobj.write_socket(conn, REQRJCT)
            elif msgtype in Clntcodes:
                if msgtype == CN_SENDCONF:
                   rval = 0
                   Clientobj.write_socket(conn, "\r\n", False)
                   break
                elif msgtype == CN_READNMLOG:
                  logwriter( "Read logfile requested")

                  mode = 0
                  if payload.find("--fullscan") != -1:
                     mode = 1
                     payload = payload.replace("--fullscan","");
                  payload = payload.strip();

                  rval      = 1
                  if os.path.exists(Workdir+"/"+"logconfig.json"):
                     with open (Workdir+"/"+"logconfig.json") as myconf_file:
                        myconfig = json.load(myconf_file)

                     if os.path.exists(payload):
                        for fobj in myconfig:
                          
                          if fobj[2] != 'null' and fobj[2] != '':
                            myfile = fobj[2]+"/"+fobj[1]
                          else:
                            myfile = fobj[1];

                          if  myfile == payload:
                            if "/etc" in myfile and "/usr/local/etc/nuhed/logs" not in myfile:
                              logwriter ("Error: Files containing */etc/* not allowed")
                              break
                            else:
                              logwriter("Processing request for file "+str(myfile))
                              rval = read_log_from_file(conn, client_sensor, myfile,mode)
                              break
                     else:
                        logwriter("Error: Requested file "+str(payload)+" not found") 
                  else:
                      logwriter("Error:Configuration file logconfig.json not found")
                    # Error occured
                  if rval == 1:
                       Clientobj.write_socket(conn, REQFAIL)
                       # Broken connection

                  break
                elif msgtype == CN_DISCONNECT_C:
                    logwriter("User: " + username + " from " + \
                               host + " disconnected")
                    break
            else: 
              # Protocol error
              Clientobj.write_socket(conn, UNKWNMSG)

    if serversocket.UseSSL:
      conn.shutdown()

    conn.close()


def do_create_poll(tid, sensor, task, rval=None):
   
    opcode = do_opcode_modify (task)
    if opcode == REQFAIL:
        return None
 
    if (create_client_message (tid, sensor, opcode, rval) > 0):
        return None
    else:
        cmsg = poll_client_message (tid, sensor)
        if not cmsg:
            return None

    return cmsg

def create_client_message(tid, sensor, msgtype, params=None):

    if not sensor:
        return 1

    if (sensor_exists(sensor) > 0):
       return 1

    # Create task entry for sensor
    task_s = [ 
               tid.getName(),
               sensor,
               msgtype,
               params,
               None,
	             0 
             ]

    Pool_t.acquireLock(MSnsrtask_l)
    Snsrtask_l.append(task_s)
    Pool_t.releaseLock(MSnsrtask_l)

    return 0
 
def poll_client_message(tid, sensor):
    out = 0
    start_t = time.time()

    # Poll Snsrtask_l
    while True:
        Pool_t.acquireLock(MSnsrtask_l)
        for x in range(len(Snsrtask_l)):
            if Snsrtask_l[x][0] == tid.getName():
                # Ok, found my entry
                if Snsrtask_l[x][4] == None:
                    # Message is not ready
                    break
                else:
                    # Message is ready
                    smsg = Snsrtask_l[x][4]
                    out = 1
                    break

        #if (round(time.time()) - \
        #    round(start_t) >= Tasktimeout):
        if Snsrtask_l[x][5] == 1:
            # Polling takes too long...
            logwriter("Sensor: " + sensor + \
                       ", current task failed, skipping")
            smsg = None
            out = 1
    
        if out:
            # Delete message entry from Snsrtask_l
            del Snsrtask_l[x]
            Pool_t.releaseLock(MSnsrtask_l)
            break
        else:
            Pool_t.releaseLock(MSnsrtask_l)
            time.sleep(Sleeptime)

    return smsg
                    
def parse_client_message(username, sensor_ac, host, message, sensor=None):

    if message[:Oplen] == CN_READNMLOG:
        args = message[Oplen:].strip()
        # User requested log output after specific timestamp
        if args != '':
            return CN_READNMLOG, args

        return CN_READNMLOG, None
    elif message[:Oplen] == CN_DISCONNECT_C:
        return CN_DISCONNECT_C, 1
    elif message[:Oplen] == CN_SENDCONF:
        if (sensor_ac.get(sensor) == "rw" or sensor_ac.get("all") == "rw"):
          payload = message[Oplen:].strip()
          fpos = open (Workdir+"/"+Logconfigjson,"w")
          fpos.write(str(payload))
          fpos.close()

          return CN_SENDCONF,None 
        else:
          logwriter("User: " + username + " from " + host + \
                    " unauthorized configuration update request for sensor " + sensor)
        return REQRJCT, 0
    else:
        return REQFAIL,0

def create_sensor_str():
    sstr = ''

    Pool_t.acquireLock(MSensor_d)
    hkeys = Sensor_d.keys()
    if len(hkeys) == 0:
        Pool_t.releaseLock(MSensor_d)
        return None;
    else:
        for hkey in hkeys:
            sstr += hkey + " "
    Pool_t.releaseLock(MSensor_d)

    return sstr.rstrip()

def sensor_exists(sname):
    Pool_t.acquireLock(MSensor_d)
    if sname in Sensor_d:
        Pool_t.releaseLock(MSensor_d)
        return 0
    else:
        Pool_t.releaseLock(MSensor_d)
        return 1

def read_log_from_file(conn, sname,fname,mode):
    #, tstamp, type=None, end=False):
    match = 0
    #fname = Alrtfile
    opcode = NC_READLOG

    pos_name_bytes = fname.encode('ascii')
    base64_bytes   = base64.b64encode(pos_name_bytes)
    pos_name       = base64_bytes.decode('ascii').replace("=","")

    posfn = Workdir+"/data/"+pos_name+".pos"
    if not os.path.exists(posfn): 
      fpos = open (posfn,"w")
      fpos.write('{"datetime":"","pos":0}')
      fpos.close()

    with open(posfn) as myjsondt:
       posdt = json.load(myjsondt)

    try:
        fp = open(fname, 'r')
    except IOError:
        logwriter("Error: IO Error")
        return 1

    if  os.fstat(fp.fileno()).st_size < posdt['pos']:
      posdt['pos'] =  0 
   
    if mode == 0:
      fp.seek(posdt['pos'])

    indx = 0
    while True:
        # Check if Shutdown flag is set
        if Shutdown == True:
            break

        slct = do_select([conn, fp], [], [], 0.1)
        if slct[0] == None:
            break

        if conn in slct[0]:
            message = Clientobj.read_socket(conn);
            if not message:
                fp.close()
                return -1

            # Client cancels alertlog reading
            if message[:3] == CN_CNCLREAD:
                Clientobj.write_socket(conn, "\r\n", False)
                break
            # Reject all other messages ....

        if fp in slct[0]:
            where = fp.tell()
            line = fp.readline()
            indx = indx + 1
            if not line:
                if line == '':
                    # EOF reached... break from loop
                    Clientobj.write_socket(conn, "\r\n", False)
                    break

                fp.seek(where)
                time.sleep(0.1)
            else:
                if match == 0:
                    line = opcode + " " + line
                Clientobj.write_socket(conn, line, False)
                match += 1
                
    posdt['pos'] = fp.tell()
    posdt['datetime'] = time.time()

    fpos = open (posfn,"w")
    fpos.write(json.dumps(posdt))
    fpos.close()
            
    fp.close()

    return 0

def passwd_authenticate_client (username, passwd):
    sensor_ac = {}
    try:
        fp = open(Passwdfile, 'r')
    except IOError as e:
        logwriter("Can't open " + Passwdfile + ", " +  e.strerror)
    else:
        lines = fp.readlines()
        fp.close()

        for i in range(len(lines)):
            split = lines[i].find( ":")
            if lines[i][:split] == username:
                substrs = lines[i].split(":")
                access = substrs[1:len(substrs)-1]
                for x in range(len(access)):
                    idx = access[x].find( ",")
                    if (idx != -1):
                        sensor_ac[access[x][:idx]] = access[x][idx+1:]
                    else:
                        return 1, None, None

                upass = base64.b64encode(hashlib.md5(passwd.encode('utf-8')).digest()).decode('utf-8')
                mpass = str(substrs[len(substrs)-1]).replace('\n','')
                if upass == mpass:
                    return 0, username, sensor_ac
                else:
                    break

    return 1, None, None


def verify_cb(conn, cert, errnum, depth, ok):
    if not Postconncheck:
        return 1

    if not ok:
        logwriter("** Can't verify certificate ** Error " + str(errnum) + " at depth " + str(depth) \
              + "\n Issuer = %s\n Subject = %s\n MD5 Digest = %s" % \
              (cert.get_issuer(), cert.get_subject(), \
               digest_cert(serversocket.crypto.dump_certificate(serversocket.crypto.FILETYPE_ASN1, cert))))
    else:
        # Check if peers FQDN is same as in certificate commonName field
        if cert.get_subject().commonName != \
           socket.getfqdn(conn.getsockname()[0]):
            # We must reject this connection
            ok = 0
            logwriter("** Can't verify certificate; peer FQDN does not match to commonName field of certificate") 
    
    return ok

def digest_cert(asn1cert):
    #digest = md5.new(asn1cert).digest()
    digest = hashlib.md5(asn1cert.encode('utf-8')).digest()
    return "%02x:" * len(digest) % tuple(map(ord, digest)).upper()[:-1]

def logwriter(information=None, level=None):
    if not information:
      return 1

    #loghdr = "[" + time.ctime(time.time()) + "][%d] " % os.getpid()
    loghdr = "[" + time.strftime("%b %d %H:%M:%S %Y", \
             time.localtime()) + "][%d] " % os.getpid()
    message = loghdr + information + "\n"

    if Logstdout == True:
        sys.stdout.write(message)
    else:
        Logfp.write(message)
        Logfp.flush()

def signal_handler_sigterm(signum, frame):
    logwriter("Caught SIGTERM, waiting for active threads to exit")
    clean_exit(0)

def clean_exit(exitstatus):
    global Shutdown

    # Set shutdown flag for active threads
    Shutdown = True
    # Shutdown threadpool
    Pool_t.stop()

    # Wait that Pool_t is set to be shutted
    while Pool_t.isStarted == True:
        pass

    logwriter("No threads executing, exiting");

    try:
        os.unlink(Nuhemgrpid)
    finally:
        sys.exit(exitstatus)

def fatalerror(error):
    logwriter("Error: " + error)
    clean_exit(1)
    
class nulldevice:
    def write(self, s):
        pass
    
def chroot_process():
    try:
        os.chroot(Chrootdir)
    except AttributeError:
        fatalerror("can't chroot() (you need python 2.2 or newer for it)")
    except OSError as e:
        fatalerror("can't chroot() to " + e.filename + ": " + e.strerror)
        
def become_daemon(fileno_l):
    try:
        pid = os.fork()
    except (OSError, AttributeError):
        fatalerror("Can't fork()")
    if pid != 0:
        sys.exit()
    else:
        os.setsid()
        if Chroot_stat == 1:
            chroot_process()
        os.chdir('/')
        sys.stdin.close()
        sys.stdout = nulldevice()
        sys.stderr = nulldevice()

        # Ensure that stray fd's are closed
        for i in range(3, 256):
            if i not in fileno_l:
                try:
                    os.close(i)
                except:
                    pass


# Globals
##########

# Variables for sensor SSL instance
Logconfigjson = ''
Slistenaddr = ''
Portsensor = 4405
Sensor_pkey = None
Sensor_cert = None

# Variables for client SSL instance
Clistenaddr = ''
Portclient = 5405
Client_pkey = None
Client_cert = None

# Client authentication method
Clientauth = 0

# Process Uid / Gid
Process_uid = None
Process_gid = None

# Min. of threads in pool
Minthreads = 5
# Max. of threads in pool
Maxthreads = 20
# Sleeping time for threads
Sleeptime = 0.1

# Tasktimeout for client polling
Tasktimeout = 60

# Shutdown flag
Shutdown = False

# Chroot status flag
Chroot_stat = 0

# Logging to file
Logstdout = False

# Daemonic mode
Daemonic = True

# Alert log
Alertlog = 1

# CA store
CAstore = None

# Post conn validation
Postconncheck = 1

Workdir = "/usr/local/nuhedlogd"
# Alert log file
Alrtfile = "/var/log/nuhedlogdalert.log"
# General log file
Logfile = "/var/log/nuhedlogd.log"

# PID file
Nuhemgrpid = "/var/run/nuhedlogd.pid"

# Configuration file
Conffile = os.path.join(Workdir,
                        "logmgrd.conf")
# User/password file
Passwdfile = os.path.join(Workdir, 
                          "logmgrusers.list")

# Opcode length
Oplen = 3

# Contains open file descriptors  
Fileno_l = []

# Fetch node uname
Nodemgrname = platform.uname()[1]

## Protocol opcodes
###################

# Protocol opcodes for sensor <-> nodemgr
SN_CLNTHELLO = "151"    # Sensor introduces to node manager
NS_CLNTHELLO = "152"    # Node manager reply
NS_ALIVE = "153"        # Hearbeat probe from node manager to sensor
SN_ALIVE = "154"        # Sensor reply
NS_REQRULES = "161"     # Node manager request rulesets
SN_REQRULES ="162"      # Sensor reply
NS_RULENAME = "163"     # Node manager request specific ruleset
SN_RULENAME = "164"     # Sensor reply
NS_SENDRULE = "165"     # Node manager uploads specific ruleset
SN_SENDRULE = "166"     # Sensor reply 
NS_REQCONF = "171"      # Node manager request sensor configuration
SN_REQCONF = "172"      # Sensor reply
NS_SENDCONF = "173"     # Node manager uploads configuration file
SN_SENDCONF = "174"     # Sensor reply
NS_REQLOG   = "175"     # Node manager request sensor logfile
SN_REQLOG   = "176"     # Sensor reply
NS_REQBLOCK   = "177"     # Node manager request sensor logfile
SN_REQBLOCK   = "178"     # Sensor reply
NS_REQACTRULES = "179"     # Node manager request sensor logfile
SN_REQACTRULES = "180"     # Sensor reply
NS_ALERTREQ = "181"     # Node manager forwards logged data to sensor
SN_ALERTREQ = "182"     # Sensor reply
SN_ALERTMSG = "191"	# Sensor sends alert message for node manager
NS_ALERTMSG = "192"	# Node manager reply
NS_RESTARTREQ = "201"   # Node manager requests sensor to restart
SN_RESTARTREQ = "202"   # Sensor reply
NS_FREEZE = "211"       # Node manager freezes managed sensor
SN_FREEZE = "212"       # Sensor reply
NS_UNFREEZE = "213"     # Node manager unfreezes managed sensor
SN_UNFREEZE = "214"     # Sensor reply
NS_REQEVENTS = "221"    # Node manager requests sensor events
SN_REQEVENTS = "222"    # Sensor_reply
NS_MODPHASE = "231"     # Node manager request to run/lock/unlock event phase
SN_MODPHASE = "232"     # Sensor reply
SN_DISCONNECT_S = "391" # Sensor disconnects from node manager
NS_DISCONNECT_S = "392" # Node manager reply
NS_DISCONNECT_N = "393" # Node manager disconnects sensor
SN_DISCONNECT_N = "394" # Sensor reply

# Protocol opcodes for client <-> nodemgr
CN_PASSWDAUTH = "101"   # Client sends auth. credentials to node manager
NC_PASSWDAUTH = "102"   # Node manager reply (succesful)
CN_QUERYSNSRS = "201"   # Client request list of managed sensors
NC_QUERYSNSRS = "202"   # Node manager reply
CN_SLCTSNSR = "203"     # Client selects sensor from list
NC_SLCTSNSR = "204"     # Node manager reply
CN_SNSRSTAT = "205"     # Client request sensor status
NC_SNSRSTAT = "206"     # Node manager reply
CN_REQRULES = "211"     # Client request active rulesets
NC_REQRULES = "212"     # Node manager reply
CN_RULENAME = "213"     # Client request specific ruleset
NC_RULENAME = "214"     # Node manager reply
CN_SENDRULE = "215"     # Client uploads specific ruleset to sensor
NC_SENDRULE = "216"     # Nodemanaged reply
CN_REQCONF = "221"      # Client request sensor configuration file
NC_REQCONF = "222"      # Node manager reply
CN_SENDCONF = "223"     # Client uploads configuration file to sensor
NC_SENDCONF = "224"     # Node manager reply
CN_REQLOG   = "225"     # Client request sensor logfile
NC_REQLOG   = "226"     # Node manage reply
CN_REQBLOCK   = "227"     # Client request sensor logfile
NC_REQBLOCK   = "228"     # Node manage reply
CN_REQACTRULES   = "229"     # Client request sensor rulesfile
NC_REQACTRULES   = "230"     # Node manage reply

CN_ALERTREQ = "231"     # Client sends logged data for sensor
NC_ALERTREQ = "232"     # Node manager reply
CN_READLOG = "241"      # Client wants to read sensor log
NC_READLOG = "242"      # Node manager starts to send sensor log
CN_READNMLOG = "243"    # Client wants to read node manager log
NC_READNMLOG = "244"    # Node manager starts to send it's log
CN_CNCLREAD = "245"     # Client cancels log reading
CN_REQEVENTS = "251"    # Client requests sensor active events
NC_REQEVENTS = "252"    # Node manager reply
CN_MODPHASE = "253"     # Client request to run/lock/unlock event phase
NC_MODPHASE = "254"     # Node manager reply
CN_RESTARTREQ = "261"	# Client request sensor to restart
NC_RESTARTREQ = "262"	# Node manager reply
CN_REMOVEREQ = "271"    # Client wants to disconnect managed sensor
NC_REMOVEREQ = "272"    # Node manager reply
CN_FREEZE = "281"	# Client wants to freeze managed sensor
NC_FREEZE = "282"       # Node manager reply
CN_UNFREEZE = "283"     # Client wants to unfreeze managed sensor
NC_UNFREEZE = "284"     # Node manager reply
CN_DELETE = "291"       # Client wants to delete sensor
NC_DELETE = "292"       # Node manager reply
CN_DISCONNECT_C = "301" # Client disconnects
NC_DISCONNECT_C = "302" # Node manager reply
NC_DISCONNECT_N = "303" # Node manager disconnects client
CN_DISCONNECT_N = "304" # Client reply

# Common opcodes
REQRJCT = "505"      # request rejected
UNKWNMSG = "507"     # unknown message
REQFAIL = "509"      # failed request

# Sensor health status
SN_ACTIVE = 0
SN_QUIET = 1
SN_RESTARTING = 2
SN_DISCONNECTED = 3
SN_NOCONNECTION = 4
SN_FREEZED = 5
SN_DEFUNC = 6

# Map protocol opcode from client to sensor
Opcmap = {
           CN_REQRULES: NS_REQRULES,
           CN_RULENAME: NS_RULENAME,
           CN_REQCONF: NS_REQCONF,
           CN_REQLOG: NS_REQLOG,
           CN_REQBLOCK: NS_REQBLOCK,
   	   CN_REQACTRULES: NS_REQACTRULES,
           CN_ALERTREQ: NS_ALERTREQ,
           CN_RESTARTREQ: NS_RESTARTREQ,
           CN_REMOVEREQ: NS_DISCONNECT_N,
           CN_SENDCONF: NS_SENDCONF,
           CN_SENDRULE: NS_SENDRULE,
           CN_FREEZE: NS_FREEZE,
           CN_UNFREEZE: NS_UNFREEZE,
	   CN_REQEVENTS: NS_REQEVENTS,
	   CN_MODPHASE: NS_MODPHASE,
           SN_REQRULES: NC_REQRULES,
           SN_RULENAME: NC_RULENAME,
           SN_REQCONF: NC_REQCONF,
           SN_REQLOG: NC_REQLOG,
           SN_REQBLOCK: NC_REQBLOCK,
           SN_REQACTRULES: NC_REQACTRULES,
           SN_ALERTREQ: NC_ALERTREQ,
           SN_RESTARTREQ: NC_RESTARTREQ,
           SN_DISCONNECT_N: NC_REMOVEREQ,
           SN_SENDCONF: NC_SENDCONF,
           SN_SENDRULE: NC_SENDRULE,
           SN_FREEZE: NC_FREEZE,
           SN_UNFREEZE: NC_UNFREEZE,
	   SN_REQEVENTS: NC_REQEVENTS,
	   SN_MODPHASE: NC_MODPHASE
         }

# Whitelist for client protocol opcodes
Clntcodes = [
              CN_QUERYSNSRS, CN_SLCTSNSR,
              CN_SNSRSTAT, CN_REQRULES,
              CN_RULENAME, CN_REQCONF,
              CN_REQLOG,CN_REQBLOCK,CN_REQACTRULES,
              CN_DISCONNECT_C, CN_DISCONNECT_N,
              CN_ALERTREQ, CN_RESTARTREQ,
              CN_REMOVEREQ, CN_READLOG,
              CN_SENDCONF, CN_SENDRULE,
              CN_READNMLOG, CN_FREEZE,
              CN_UNFREEZE, CN_DELETE,
	      CN_REQEVENTS, CN_MODPHASE
            ]

# Sensor status strings
Statmap = {
            SN_ACTIVE: "ACTIVE",
            SN_QUIET: "QUIET",
            SN_RESTARTING: "RESTARTING",
            SN_DISCONNECTED: "DISCONNECTED",
            SN_NOCONNECTION: "BAD DISCONNECT",
            SN_FREEZED: "FREEZED",
            SN_DEFUNC: "DEFUNC"
          }

# Dictionary for registered sensors
Sensor_d = {}
# Task list for sensors
Snsrtask_l = []

# Initialization and validation before start
############################################

# Fetch command line parameters.
try:
    opts, args = getopt.getopt(sys.argv[1:], 'c:rfhv',\
    ['help', 'version'])
except:
    sys.stderr.write("Try 'nuhemgrd.py --help' for more information.\n")
    sys.exit(1)

for o, a in opts:
    if o == '-c':
        Conffile = a
    if o == '-r':
        Logstdout = True
    if o == '-f':
        Daemonic = False
    if o == '--help' or o == '-h':
        print (__doc__)
        sys.exit()
    if o == '--version' or o == '-v':
        print ("Lognode manager v. 1.0.1")
        sys.exit()

if (Logstdout == True and Daemonic == True):
    sys.stderr.write("Error: you can only specify stdout logging (-r) with foreground mode (-f)\n")
    sys.exit(1)

# Check SSL support
if serversocket.UseSSL == -1:
    sys.stderr.write("Error: can't find pyOpenSSL module\n")
    sys.exit(1)

try:
    fp = open(Conffile, 'r')
except IOError:
    sys.stderr.write("Error: can't open node manager configuration file\n")
    sys.exit(1)
else:
    clines = fp.readlines()
    fp.close()

    for i in range(len(clines)):
        if "#" in clines[i]:
           continue

        # Sensor socket values
        if (clines[i].find("Slistenaddr") != -1):
            m = clines[i].find( " ")
            Slistenaddr = clines[i][m:-1].strip()
        elif (clines[i].find( "Portsensor") != -1):
            m = clines[i].find( " ")
            Portsensor = clines[i][m:-1].strip()
        elif (clines[i].find( "Sensorkey") != -1):
            m = clines[i].find( " ")
            Sensor_pkey = clines[i][m:-1].strip()
        elif (clines[i].find( "Sensorcrt") != -1):
            m = clines[i].find( " ")
            Sensor_cert = clines[i][m:-1].strip()
        # Client socket values
        elif (clines[i].find( "Clistenaddr") != -1):
            m = clines[i].find( " ")
            Clistenaddr =  clines[i][m:-1].strip()
        elif (clines[i].find( "Portclient") != -1):
            m = clines[i].find( " ")
            Portclient = clines[i][m:-1].strip()
        elif (clines[i].find( "Clientkey") != -1):
            m = clines[i].find( " ")
            Client_pkey = clines[i][m:-1].strip()
        elif (clines[i].find("Clientcrt") != -1):
            m = clines[i].find( " ")
            Client_cert =  clines[i][m:-1].strip()
        # General values
        elif (clines[i].find( "Logconfigjson") != -1):
            m = clines[i].find( " ")
            Logconfigjson = clines[i][m:-1].strip()
        elif (clines[i].find( "Name") != -1):
            m = clines[i].find( " ")
            Nodemgrname = clines[i][m:-1].strip()
        elif (clines[i].find( "Uid") != -1):
            m = clines[i].find( " ")
            Process_uid = clines[i][m:-1].strip()
        elif (clines[i].find( "Gid") != -1):
            m = clines[i].find( " ")
            Process_gid = clines[i][m:-1].strip()
        elif (clines[i].find( "Minthreads") != -1):
            m = clines[i].find( " ")
            Minthreads = clines[i][m:-1].strip()
        elif (clines[i].find( "Maxthreads") != -1):
            m = clines[i].find( " ")
            Maxthreads = clines[i][m:-1].strip()
        elif (clines[i].find( "Tasktimeout") != -1):
            m = clines[i].find( " ")
            Tasktimeout = clines[i][m:-1].strip()
        elif (clines[i].find( "Logdest") != -1):
            m = clines[i].find( " ")
            Alertlog = clines[i][m:-1].strip()
        elif (clines[i].find( "CAfile") != -1):
            m = clines[i].find( " ")
            CAstore = clines[i][m:-1].strip()
        elif (clines[i].find( "Postconncheck") != -1):
            m = clines[i].find( " ")
            Postconncheck = clines[i][m:-1].strip()
        elif (clines[i].find( "Clienauth") != -1):
            m = clines[i].find( " ")
            tmpstr = clines[i][m:-1].strip()
            if tmpstr == "passwd":
                Clientauth = 0
            elif tmpstr == "pubkey":
                Clientauth = 1
            else:
               sys.stderr.write("Error: Clientauth must be passwd or pubkey\n");
               sys.exit(1)

if (Sensor_pkey == None):
    sys.stderr.write("Error: you must have private key for sensor instance\n");
    sys.exit(1)
if (Client_pkey == None):
    sys.stderr.write("Error: you must have private key for client instance\n");
    sys.exit(1)
if (Sensor_cert == None):
    sys.stderr.write("Error: you must have certificate for sensor instance\n");
    sys.exit(1)
if (Client_cert == None):
    sys.stderr.write("Error: you must have certificate for client instance\n");
    sys.exit(1)


if (Slistenaddr == Clistenaddr and Portsensor == Portclient):
    sys.stderr.write("Error: you must specify different address:port pair for sensor and client instances\n");
    sys.exit(1)

# Create server object for sensors
if serversocket.UseSSL:
    # Checks for sensor instances keys and certs
    if os.access(Sensor_pkey, os.R_OK) == 0:
        sys.stderr.write("Error: " + Sensor_pkey + " doesn't exists! You must have private key and"\
               + " certificate for SSL server.\n")
        sys.exit(1)
    if os.access(Sensor_cert, os.R_OK) == 0:
        sys.stderr.write("Error: " + Sensor_cert + " doesn't exists! You must have private key and"\
               + " certificate for SSL server.\n")
        sys.exit(1)
    # Checks for client instances keys and certs
    if os.access(Client_pkey, os.R_OK) == 0:
        sys.stderr.write("Error: " + Client_pkey + " doesn't exists! You must have private key and"\
               + " certificate for SSL server.\n")
        sys.exit(1)
    if os.access(Client_cert, os.R_OK) == 0:
        sys.stderr.write("Error: " + Client_cert + " doesn't exists! You must have private key and"\
               + " certificate for SSL server.\n")
        sys.exit(1)

    # Create server instances
    Sensorobj = serversocket.SSLServer(Sensor_pkey, Sensor_cert, True)
    Clientobj = serversocket.SSLServer(Client_pkey, Client_cert, True)

    # Authenticate sensors
    Sensorobj.set_verify(serversocket.SSL.VERIFY_PEER or \
                         serversocket.SSL.VERIFY_FAIL_IF_NO_PEER_CERT, verify_cb, CAstore)

# Check Portsensor
if type(Portsensor) == type(''):
    try:
        Portsensor = int(Portsensor)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Portsensor\n")
       sys.exit(1)

# Check Portclient
if type(Portclient) == type(''):
    try:
        Portclient = int(Portclient)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Portclient\n")
       sys.exit(1)

# Check Process_uid
if (type(Process_uid) == type('')):
    try:
        Process_uid = int(Process_uid)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Uid\n")
       sys.exit(1)

# Check Process_gid
if (type(Process_gid) == type('')):
    try:
        Process_gid = int(Process_gid)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Gid\n")
       sys.exit(1)

# Check Minthreads
if (type(Minthreads) == type('')):
    try:
        Minthreads = int(Minthreads)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Minthreads\n")
       sys.exit(1)

# Check Maxthreads
if (type(Maxthreads) == type('')):
    try:
        Maxthreads = int(Maxthreads)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Maxthreads\n")
       sys.exit(1)

# Check Sleeptime
if (type(Sleeptime) == type('')):
    try:
        Sleeptime = float(Sleeptime)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Sleeptime\n")
       sys.exit(1)

# Check Tasktimeout
if (type(Tasktimeout) == type('')):
    try:
        Tasktimeout = int(Tasktimeout)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Tasktimeout\n")
       sys.exit(1)

# Check Alertlog
if (type(Alertlog) == type('')):
    try:
        Alertlog = int(Alertlog)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Alertlog\n")
       sys.exit(1)

if Clientauth == 1:
    sys.stderr.write("Error: only passwd is supported for Clientauth at the moment\n")
    sys.exit(1)

if os.access(Passwdfile, os.R_OK) == 0:
    sys.stderr.write("Warning: can't open " + Passwdfile + " for reading\n")

# Check Postconncheck
if (type(Postconncheck) == type('')):
    try:
        Postconncheck = int(Postconncheck)
    except ValueError:
       sys.stderr.write("Error: invalid specification, Postconncheck\n")
       sys.exit(1)

# Check if nuhemgrd is already running
if os.path.exists(Nuhemgrpid):
    sys.stderr.write("Error: PID file exists (" + Nuhemgrpid + ")\n")
    sys.exit(1)

# Set default umask for Nuhe files
os.umask(0o0066)

# Open logging file
if Logstdout == False:
    try:
        Logfp = open(Logfile, 'a')
    except IOError as e:
        sys.stderr.write("Error: Can't open file, " + Logfile \
                          + " for logging: " + e.strerror + "\n")
        sys.exit(1)
    Fileno_l.append(Logfp.fileno())

# Open alertlog
if Alertlog == 1:
    try:
        Alrfp = open(Alrtfile, 'a')
    except IOError as e:
        sys.stderr.write("Error: Can't open file, " + Alrtfile \
                          + " for logging: " + e.strerror + "\n")
        sys.exit(1)
    Fileno_l.append(Alrfp.fileno())

if Daemonic == True:
    # Daemonize
    sys.stdout.write("Forking to background\n")
    become_daemon(Fileno_l)

# Set signal handlers
signal.signal(signal.SIGTERM, signal_handler_sigterm)
if Daemonic == False:
    # Catch SIGINT when running on foreground
    signal.signal(signal.SIGINT, signal_handler_sigterm)

try:
    fp = open(Nuhemgrpid, "w")
except IOError:
    fatalerror("can't write pid file to disk")

my_pid = os.getpid()
fp.write(str(my_pid) + "\012")
fp.close()

# Create threadpool
Pool_t = ThreadPool.ThreadPool(Minthreads, \
                               Minthreads, \
                               Maxthreads)

# Create lock object to protect Server_d
MSensor_d = Pool_t.Lock()
# Create lock object to protect Snsrtask_l
MSnsrtask_l = Pool_t.Lock()

# Start threadpool
Pool_t.start()
        
# Start work
wait_connection()
