# Copyright (C) 2003,2007 Tuomo Makinen (tjam@users.sourceforge.net)
# Licensed under the terms of the GNU General Public License
# serversocket.py v. 0.0.3

import os
try:
    from OpenSSL import SSL, crypto
except ImportError:
    print ("FAILED TO IMPORT SSL")
    UseSSL = -1
else:
    UseSSL = 1
    
from errno import EINTR
import socket

RECVBUFSIZE = 4096
CRLF = "\r\n"
BACKLOG = 5
CIPHER_LIST = "ALL:!ADH:!LOW:!EXP:!MD5:@STRENGTH"

DEBUG = 1

class Server:
    """ Base class for SSLServer and RegularServer classes."""
    def __init__(self):
    	self.delimiter = CRLF
    	self.recvsize = RECVBUFSIZE
    	self.backlog = BACKLOG

    def activate_server(self, server, port):
        try:
            self.sockobj.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            self.sockobj.bind((server, port))
            self.sockobj.listen(self.backlog)
        except socket.error as e:
            return (-1, e[1])
        return(0,)

    def accept_conn(self):
        return self.sockobj.accept()
        
    def close_socket(self):
        self.sockobj.close()

    def set_delimiter(self, delimiter):
        self.delimiter = delimiter

    def get_delimiter(self):
        return self.delimiter

    def set_recvsize(self, recvbuf):
        self.recvsize = recvbuf

    def get_recvsize(self):
        return self.recvsize

    def set_backlog(self, backlog):
        self.backlog = backlog

    def get_backlog(self):
        return self.backlog

class SSLServer(Server):
    def __init__(self, pkey, cert, threaded=False):
        # Run parent constructor
        Server.__init__(self)
        self.__threaded = threaded

        # Initialize SSL context
        self.ctx = SSL.Context(SSL.SSLv23_METHOD)
        self.ctx.set_options(SSL.OP_NO_SSLv2)
        self.ctx.set_cipher_list(CIPHER_LIST)
        self.ctx.use_privatekey_file(pkey)
        self.ctx.use_certificate_file(cert)

    def set_verify (self, options, cb_func, ca_certs):
        self.ctx.set_verify (options, cb_func)
        try:
            self.ctx.load_verify_locations(ca_certs)
        except SSL.Error:
            # Can't load trusted CA certificates ...
            pass

    def do_socket(self):
        if self.__threaded:
            # We want to be thread safe
            self.sockobj = SSL.Connection(self.ctx, socket.socket(socket.AF_INET, socket.SOCK_STREAM))
            #self.sockobj = tsafe.Connection(self.ctx, socket.socket(socket.AF_INET, socket.SOCK_STREAM))
        else:
            self.sockobj = SSL.Connection(self.ctx, socket.socket(socket.AF_INET, socket.SOCK_STREAM))
        
    def read_socket(self, conn):

        data = ''
        while 1:
            try:
                chunk = conn.recv(self.recvsize)
            except (SSL.WantReadError, SSL.WantWriteError):
                if DEBUG:
                  print ("** SSL.WantReadError, SSL.WantWriteError")
                continue
            except SSL.ZeroReturnError:
                if DEBUG:
                  print ("** SSL.ZeroReturnError")
                chunk = None
            except SSL.Error:
                if DEBUG:
                  print ("** SSL.Error")
                chunk = None
            if not chunk:
                return
            else:
                data = data + str(chunk.decode('utf-8'))
                if data[-2:] == self.delimiter:
                   return clean_message(data, self.delimiter)

                return False
                   
    def write_socket(self, conn, message, delim=True):
        if delim == True:
            message = message + self.delimiter

        totalsent = 0
        while totalsent < len(message):
            try:
                sent = conn.send(message[totalsent:])
            except (SSL.WantReadError, SSL.WantWriteError):
                continue
            except SSL.ZeroReturnError:
                break
            except SSL.Error:
                break
            totalsent = totalsent + sent
            
class RegularServer(Server):
    def __init__(self):
        # Run parent constructor
        Server.__init__(self)

    def do_socket(self):
        self.sockobj = socket.socket(socket.AF_INET, socket.SOCK_STREAM)

    def read_socket(self, conn):
        data = ''
        while 1:
            try:
                data = conn.recv(self.recvsize)
            except socket.error as e:
                data = None
            if not data:
                return
            elif data[-2:] == CRLF:
                return clean_message(data, self.delimiter)

    def write_socket(self, conn, message):
        message = message + CRLF
        totalsent = 0
        while totalsent < len(message):
            try:
                sent = conn.send(message[totalsent:])
            except socket.error as e:
                break
            totalsent = totalsent + sent

def clean_message(message, delimiter):
    if len(message) >= 2:
        if message[-2:] == delimiter:
            message = message[:-2]
        elif message[-1:] in delimiter:
            message = message[:-1]
    return message
