/* nuheclient.c
 *
 * Copyright (C) 2025 Tuomo Makinen (tjam@users.sourceforge.net)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */


#include "nuheclient.h"

/* Timeout in seconds between connection retry */
#define CONN_RETRY_TIMEOUT 5

static int open_connection (struct internal *vars, SSL_CTX *ctx);
static int handle_connection (SSL * ssl, struct internal *vars);
static enum msgtype get_message_opcode (char *msg, struct internal *vars);

static char *get_prog_name_from_path (char *progpath);
static int is_cmd_whitelisted(char *cstring,char *whitelistfile);
static char *exec_sensor_cmd_op (enum msgtype type, char *ruledir,char *payload,char *whitelistfile);
static char *exec_sensor_file_op (char *param, enum msgtype type, char *payload, char *offset_container_file);
static void get_message_payload (char *srcbuf, char *dstbuf, unsigned int dstlen);
static int handle_operation (SSL * ssl, char *param, enum msgtype type, char *payload, char *whitelistfile,char *offset_container_file);
static int send_pending_messages (SSL * ssl);
static int save_sensor_file (char *cfile, char *fcontents);
static int alert_failsafe_write (const char *msg);
static int move_alerts_to_failsafe ();
void sigaction_handler(int signo);

/* Mutex for msglist */
static pthread_mutex_t msglist_m = PTHREAD_MUTEX_INITIALIZER;

/* list_s for alert messages */
static struct list_s *msglist;

/* Holds original status of Nuhe */
static enum status orig_state;

/* Are we connected ? */
static unsigned int connected = 0;

/* Protocol phases */
enum cphase
{
  STARTCOM = 0,
  CONCHECK
};

/* Protocol message types */
enum msgtype
{
  CLNTHELLO = 0,
  SRVRHELLO,
  SRVRPING,
  SRVRREQRS,
  SRVRGETRN,
  SRVRSENDRN,
  SRVRREQCN,
  SRVRSENDCN,
  SRVRREQLOG,
  SRVRSENDLOG,
  SRVRSENDCMD,
  SRVRREQRULES,
  SRVRREQACTRULES,
  SRVRREQPIDSTATUS,
  SRVRREQALRT,
  SRVRREQBOOT,
  SRVRREQQUIT,
  SRVRREQFRZ,
  SRVRREQUNFRZ,
  SRVRREQEVNTS,
  SRVRMODPHASE,
  REQRJCT,
  UNKNOWN
};

void sigaction_handler(int signo){


}


void
nuhe_connector (struct internal *vars)
{
  SSL_CTX *ctx;
  struct timeval timeout;
  int ci;

  struct sigaction sigact;
  memset(&sigact, 0, sizeof(sigact));
  sigact.sa_handler = sigaction_handler;
  sigact.sa_flags   = SA_NOCLDWAIT;

  if (sigaction(SIGCHLD, &sigact, NULL) < 0)
    printf("Sigaction reg is failed\n");

  orig_state = vars->run_stat;
  ci = 0;

  /* Initialize mutex & cv for FREEZE state */
  pthread_mutex_init (&freeze_sync.state_mutex, NULL);
  pthread_cond_init (&freeze_sync.state_cv, NULL);

  /* Create list_s for messages */
  msglist = slist_create ();
  if (msglist == NULL)
    {
      perror ("malloc() failed");
      exit (1);
    }

  if (init_openssl () > 0)
    logwriter
      ("Error: OpenSSL initialization failed, node manager connector can't be started!");

  else
    {
      /* Create SSL context, CA directory is currently NULL */
      ctx = create_ssl_context (vars->certificate, vars->privatekey, vars->cafile, NULL);
      if (ctx == NULL)
        logwriter
          ("Error: OpenSSL SSL_CTX creation failed, node manager connector can't be started!");

      else
        {
          logwriter ("Nuhe node manager connector started");
          while (1)
	    {
              connected = 0;

	      if (open_connection (vars, ctx) < 0)
	        {
	          /* bail out ? */
                  if (vars->connretries > 0 && ci >= vars->connretries)
		    {
		      logwriter
		        ("Error: connection to node manager failed, giving up");
                      /* Move alerts to failsafe store */
                      move_alerts_to_failsafe ();
                      break;
		    }
	          /* Something went wrong ...
	           * wait for CONN_RETRY_TIMEOUT and try again ...
	           */
	          timeout.tv_sec = CONN_RETRY_TIMEOUT;
	          timeout.tv_usec = 0;

	          select (0, NULL, NULL, NULL, &timeout);
	          ci++;
	        }

              /* Move alerts to failsafe store */
              move_alerts_to_failsafe ();

	      /* Check if Nuhe has received SIGTERM, SIGINT or
	       * SIGHUP signal
	       */
	      if (quit_flag != RUNNING)
	        break;
	    }
          SSL_CTX_free (ctx);
        }
    }

  /* Delete msglist and associated mutex & condition variable */
  slist_delete (msglist);
  msglist = NULL;

  pthread_mutex_destroy (&freeze_sync.state_mutex);
  pthread_cond_destroy (&freeze_sync.state_cv);
}

static int
open_connection (struct internal *vars, SSL_CTX *ctx)
{
  SSL *ssl;

  long err;
  int closedc;
  int ret;

  char hconn[STRBUF] = { 0 };

  closedc = 0;
  snprintf (hconn, sizeof (hconn), "%s:%d", vars->manageraddr,
	    vars->managerport);

  seed_prng (1024);

  /* Create SSL connection to node manager */
  ssl = create_ssl_connection (ctx, hconn);
  if (ssl == NULL)
    return -1;

  /* Initiate handshake */
  ret = SSL_connect (ssl);
  if (ret <= 0)
    {
      int ec;
      char estr[STRBUF] = { 0 };

      ec = SSL_get_error (ssl, ret);
      if (ec == SSL_ERROR_SYSCALL)
	snprintf (estr, sizeof (estr),
		  "Error: TLS/SSL handshake was not successful, %s",
		  strerror (errno));
      else
        {
          char err[128] = { 0 };

          ERR_error_string (ERR_get_error(), err);
	  snprintf (estr, sizeof (estr),
	       	    "Error: TLS/SSL handshake was not successful, %s",
                     err);
        }

      logwriter (estr);
closedc = 1;
goto out;
    }

  /* Post connection validation */
  if ((err = post_conn_check (ssl, vars->manageraddr)) != X509_V_OK)
    {
      char data[STRBUF] = { 0 };

      snprintf(data, sizeof (data), "Error: peer certificate, %s",
               X509_verify_cert_error_string (err));
      logwriter (data);
      closedc = 1;
      goto out;
    }

  /* Handle our connection */
  if (handle_connection (ssl, vars) < 0)
    {
      /* Peer probably closed socket */
      logwriter
	("Error: node manager shutted connection, retrying to connect");
      closedc = 1;
    }

out:
  SSL_shutdown (ssl);
  SSL_free (ssl);

  if (closedc > 0)
    return -1;

  return 0;
}

static int
handle_connection (SSL * ssl, struct internal *vars)
{
  char txbuf[BUFSIZE] = { 0 };
  char managername[STRBUF] = { 0 };
  char *rxbuf;

  unsigned int rval;
  unsigned int darr_sz;
  unsigned int i;
  int err;
  int sock;

  enum cphase task;
  struct rulefiles *rstrc;
  struct timeval timeout;
  fd_set fds;

  /* Initial protocol phase */
  task = STARTCOM;

  /* Read channel fd for ssl */
  sock = SSL_get_rfd (ssl);

  /* We're connected */
  connected = 1;

  while (1)
    {

      /* quit_flag is set to exit state */
      if (quit_flag != RUNNING)
	{
          /* Make sure that FREEZE state threads are signaled; they
           * should be able to perform clean-up tasks before exit
           */
          pthread_mutex_lock (&freeze_sync.state_mutex);
          vars->run_stat = orig_state;
          pthread_cond_broadcast (&freeze_sync.state_cv);
          pthread_mutex_unlock (&freeze_sync.state_mutex);

          /* Make sure all messages has been sent to node manager */
          send_pending_messages (ssl);

          if (quit_flag == QUIT)
            write_sock (ssl, "391 BYE\r\n");
          else if (quit_flag == SHUTDOWN)
            write_sock (ssl, "394 BYE\r\n");
          else if (quit_flag == RESTART)
            write_sock (ssl, "202 RESTARTING\r\n");
	  return 0;
	}

      if (task == STARTCOM)
	{
	  /* Let's say hello to server */
	  if (vars->run_stat != DEFUNC)
	    snprintf (txbuf, sizeof (txbuf), "151 %s %s HELLO\r\n",
		      vars->sensorname, VERSION);
	  else
	    /* DEFUNC state hello */
            snprintf (txbuf, sizeof (txbuf), "151 %s %s DEFUNC\r\n",
	              vars->sensorname, VERSION);

	  if (write_sock (ssl, txbuf) < 1)
	    {
              enum msgtype opcode;

	      rval = read_sock (ssl, &rxbuf);

	      if (rval != 0){
		return -1;
               }
	      opcode = get_message_opcode (rxbuf, vars);
	      if (opcode == SRVRHELLO)
		{
                  char *errmsg;

		  /* Fetch node manager name */
		  get_message_payload (rxbuf, managername, STRBUF);
		  if (managername[strlen (managername) - 1] == '\n')
		    {
		      managername[strlen (managername) - 1] = '\0';
		    }

		  logwriter ("Connection to node manager established");

                  /* Report errors and warnings to node manager 
                   ********************************************
                   */

                  errmsg = (char *) calloc (LOGBUF, sizeof (char));
                  if (errmsg == NULL)
                    {
                      logwriter("CRITICAL: allocating space for errmsg failed, calloc()");
                    }
                  else
                    {
                      /* We are running in DEFUNC state; report error to node manager*/
                      if (vars->run_stat == DEFUNC)
                        {
                          snprintf (errmsg, LOGBUF, "[ERROR] %s, %s",
                                    vars->sensorname, vars->errstr);
                          send_alarm ((const char *) errmsg);
                        }

                      /* Check if we are running with .backup configuration file 
                       * and report it to node manager
                       */
                      if (strstr (vars->cfile, ".backup") != NULL)
                        {
                          snprintf (errmsg, LOGBUF,
                                    "[WARNING] %s, running with configuration file backup",
                                    vars->sensorname);
                          send_alarm ((const char *) errmsg);
                        }

                      /* We're running with file monitor disabled; report it no node manager */
                      if (vars->active_fmon < 1)
                        {
                          snprintf (errmsg, LOGBUF,
                                    "[WARNING] %s, running with file monitor disabled",
                                    vars->sensorname);
                          send_alarm ((const char *) errmsg);
                        }

                      free (errmsg);
                    }

                  /* Handshake is done, move to CONCHECK mode */
                  task = CONCHECK;
		}
              free (rxbuf);
	    }
	  else
	    return -1;
	}
      else if (task == CONCHECK)
	{
	  /* This is so called idle mode; If select timeouts we
           * send pending alert messages for node manager, otherwise
           * we listen ALIVE messages from node manager, reply to them
           * and switch between tasks
	   */

          enum msgtype task_n;

	  FD_ZERO (&fds);
	  FD_SET (sock, &fds);

	  timeout.tv_sec = 1;
	  timeout.tv_usec = 0;

	  err = select (sock + 1, &fds, NULL, NULL, &timeout);

	  /* select timeout; send pending alarms to node manager */
	  if (err == 0)
	    {
	      if (send_pending_messages (ssl) < 0)
                return -1;
	      continue;
	    }

	  /* select error */
	  if (err == -1)
	    continue;

	  /* We are ready to read */
	  rval = read_sock (ssl, &rxbuf);

	  if (rval != 0){
	    return -1;
	  }

	  task_n = get_message_opcode (rxbuf, vars);
          /* free rxbuf for all other tasks */
          if (task_n == SRVRGETRN    || task_n == SRVRREQALRT ||
              task_n == SRVRSENDCN   || task_n == SRVRSENDRN  ||
              task_n == SRVRMODPHASE || task_n == SRVRREQLOG || 
	      task_n == SRVRSENDLOG  || task_n == SRVRREQACTRULES ||
	      task_n == SRVRSENDCMD  || task_n == SRVRREQPIDSTATUS)
            { }
          else
            {
              if (rxbuf != NULL)
                free (rxbuf);
            }

	  if (task_n == SRVRPING)
	    {
	      /* Send ALIVE reply to server */
	      if (write_sock (ssl, "154 OK\r\n") < 1)
		continue;
	      else
		return -1;
	    }
          else if (task_n == SRVRREQQUIT)
            {
              logwriter("Node manager requested shutdown, exiting...");
              quit_flag = SHUTDOWN;
              continue;
            }
          else if (task_n == SRVRREQBOOT)
            {
              logwriter("Node manager requested restart, restarting...");
              quit_flag = RESTART;
              vars->run_stat_mgr = RESTARTREQ;
              continue;
            }
          else if (task_n == SRVRREQFRZ && vars->run_stat != FREEZE)
            {
              vars->run_stat = FREEZE;
              logwriter("Node manager requested freezing, sensor stopped");
              if (write_sock (ssl, "212 FREEZED\r\n") > 0)
                return -1;
              continue;
            }
          else if (task_n == SRVRREQUNFRZ)
            {
              /* Change state back to RUNNING and signal waiting threads */
	      pthread_mutex_lock (&freeze_sync.state_mutex);
              vars->run_stat = orig_state;
	      pthread_cond_broadcast (&freeze_sync.state_cv);
	      pthread_mutex_unlock (&freeze_sync.state_mutex);
           
              logwriter("Node manager requested unfreezing, sensor running");
              if (write_sock (ssl, "214 UNFREEZED\r\n") > 0)
                return -1;
              continue; 
            }
	  else if (task_n == SRVRREQRS)
	    {
	      memset (txbuf, 0, sizeof (txbuf));
	      snprintf (txbuf, sizeof (txbuf), "162 ");
	      darr_sz = d_array_get_size (vars->rules);
	      for (i = 0; i < darr_sz; i++)
		{
		  rstrc =
		    (struct rulefiles *) vars->rules->element[i]->payload;
		  strncat (txbuf, rstrc->fname, sizeof (txbuf));
		  if (i + 1 < darr_sz)
		    strncat (txbuf, " ", sizeof (txbuf));
		}
	      strncat (txbuf, CRLF, sizeof (txbuf));

	      /* Send reply to node manager */
	      if (write_sock (ssl, txbuf) < 1)
		task = CONCHECK;
	      else
		return -1;
	    }
	  else if (task_n == SRVRGETRN)
	    {  
              int out;
	      char rulefile[STRBUF] = { 0 };

              out = 0;
	      get_message_payload (rxbuf, rulefile, STRBUF);
              free (rxbuf);

	      darr_sz = d_array_get_size (vars->rules);
	      for (i = 0; i < darr_sz; i++)
		{
		  rstrc =
		    (struct rulefiles *) vars->rules->element[i]->payload;
		  if (!strcmp (rulefile, rstrc->fname))
		    {
		      if (handle_operation (ssl, rstrc->fpath, task_n,NULL,NULL,NULL) <
			  0)
			return -1;
                      out = 1;
		      break;
		    }
		}
              if (!out)
                {
                  if (write_sock (ssl, "509 REQUEST FAILED\r\n") > 0)
                    return -1;
                }
	      task = CONCHECK;
	    }
	  else if (task_n == SRVRREQCN)
	    {
	
	      if (handle_operation (ssl, vars->cfile, task_n,NULL,NULL,NULL) < 0)
		return -1;
              task = CONCHECK;
	    }
	  else if (task_n == SRVRSENDLOG)
            {
              char *payload = (char *) calloc (strlen (rxbuf) + 1, sizeof (char));
              if (payload == NULL)
                {
                  logwriter("CRITICAL: allocating space for logfiles failed, calloc()");
                  return 1;
                }

              get_message_payload (rxbuf, payload, strlen (rxbuf));
              free (rxbuf);

	      FILE *fpl;
              char *mytoken = strtok(payload, ";");
              fpl  = fopen (vars->logfilesfile, "w");

              if (fpl == NULL){
		 logwriter("CRITICAL: Failed to open logfilesfile for write");
                 return -1;
              }
              while( mytoken != NULL ) {
                fprintf(fpl, "%s \r\n", mytoken ); 
                mytoken = strtok(NULL, ";");
              }
              fclose (fpl);
              if (write_sock (ssl, "402 OK\r\n") > 0)
                return -1;

              task = CONCHECK;
            }
          else if (task_n == SRVRREQLOG)
            { 
              char datefilter[STRBUF] = { 0 };
	      get_message_payload (rxbuf, datefilter, STRBUF);
              free(rxbuf);
	      if (handle_operation (ssl, vars->logfile, task_n,datefilter,NULL,"/usr/local/etc/nuhed/logoffset.dat") < 0){
                return -1;
	      }

              task = CONCHECK;
            }
          else if (task_n == SRVRREQACTRULES)
            {
              if (handle_operation (ssl, vars->rulesfile, task_n,NULL,NULL,NULL) < 0)
                return -1;
              task = CONCHECK;
            }
          else if (task_n == SRVRREQPIDSTATUS)
            {
	      FILE *fp;
              char procpath[STRBUF] = { 0 };
              char *payload = (char *) calloc (strlen (rxbuf) + 1, sizeof (char));
              if (payload == NULL)
                {
                  logwriter("CRITICAL: allocating space failed, calloc()");
                  return 1;
                }
	      get_message_payload (rxbuf, payload, strlen(rxbuf));
              free(rxbuf);

              char *my_cr = strstr(payload, "\n");
              if (my_cr != NULL)
                 payload[ strlen(payload) - 1 ] = '\0';
              char *my_ret = strstr(payload, "\r");
              if (my_ret != NULL)
                 payload[ strlen(payload) - 1 ] = '\0';
              
	      if (strlen(payload) == 0 ){
                if (write_sock (ssl, "509 INV_PID_RECEIVED\r\n") > 0)
                  return -1;
	      }else{
                snprintf(procpath, sizeof(procpath), "%s%s", "/proc/", payload);
   	        fp = fopen (procpath, "r");
                if (fp == NULL){
                  if (write_sock (ssl, "194 0\r\n") > 0)
	            return -1;
	        }else{
	          fclose(fp);
                  if (write_sock (ssl, "194 1\r\n") > 0)
                    return -1;
	        }
	      }
              task = CONCHECK;
            }
          else if (task_n == SRVRSENDCMD)
            {
              char *payload = (char *) calloc (strlen (rxbuf) + 1, sizeof (char));
              if (payload == NULL)
                {
                  logwriter("CRITICAL: allocating space for logfiles failed, calloc()");
                  return 1;
                }

	      get_message_payload (rxbuf, payload, strlen(rxbuf));
              free(rxbuf);
	      if (strcmp(payload, "") != 0){
                    if (handle_operation (ssl, vars->ruledir, task_n,payload,vars->whitelistfile,NULL) < 0){
                      return -1;
		    }
	      }

              task = CONCHECK;
            }

          else if (task_n == SRVRSENDCN || task_n == SRVRSENDRN)
            {
              char *fname;
              char *fcontents;
              char *payload;
              int rval;


              payload = (char *) calloc (strlen (rxbuf) + 1, sizeof (char));
              if (payload == NULL)
                {
                  logwriter("CRITICAL: allocating space for rule/config failed, calloc()");
                  return 1;
                }

              get_message_payload (rxbuf, payload, strlen (rxbuf));
              free (rxbuf);

              /* If task_n is SRVRSENDRN, we must extract
               * ruleset name from message
               */
              if (task_n == SRVRSENDRN)
                {
                  char *rnptr;
                  char fpath[STRBUF] = { 0 };

                  /* adjust rnptr to message payload */
                  rnptr = &payload[0];
                  while (!isspace ((int) *rnptr))
                    rnptr++;
                  *rnptr = '\0';

                  snprintf (fpath, sizeof (fpath), "%s/%s", vars->ruledir, payload);
                      
                  fname = &fpath[0];
                  fcontents = ++rnptr;
                }
              else
                {
                  fname = vars->cfile;
                  fcontents = payload;
                }

              /* Save configuration/ruleset file */
              rval = save_sensor_file (fname, fcontents);
              free (payload);

              if (rval != 0)
                {
                  if (write_sock (ssl, "509 REQUEST FAILED\r\n") > 0)
                    return -1;
                }
              else
                {
                  if (write_sock (ssl, (task_n == SRVRSENDCN ? 
                                  "174 OK\r\n" : "166 OK\r\n")) > 0)
                    return -1;
                }
              task = CONCHECK;
            }
	  else if (task_n == SRVRREQALRT)
	    {
              char *alertstr;

	      if (vars->manageralrt > 0)
		{
                  alertstr = (char *) calloc (strlen (rxbuf) + 1, sizeof (char));
                  if (alertstr == NULL)
                    {
                      free (rxbuf);
                      logwriter("CRITICAL: allocating alarm event failed, calloc()");
                      return 1;
                    }

		  /* Get payload from message */
		  get_message_payload (rxbuf, alertstr, BUFSIZE);
                  free (rxbuf);

		  /* Process alert message */
		  process_alert_data (vars, managername, alertstr);
                  free (alertstr);

		  if (write_sock (ssl, "182 OK\r\n") > 0)
		    return -1;
		  task = CONCHECK;
		}
	      else
		{
                  free (rxbuf);
		  if (write_sock (ssl, "505 REQUEST REJECTED\r\n") > 0)
		    return -1;
		  task = CONCHECK;
		}
	    }
	  else if (task_n == SRVRREQEVNTS)
	    {
              struct d_array *earr;

              earr = d_array_create ();
              if (earr == NULL)
	        {
                  logwriter("CRITICAL: allocating d_array for pending events failed, malloc()");
                  return 1;
                }

              /* Collect all active events */
              if (collect_pending_events (vars, earr) < 1)
                {
                  if (d_array_get_size (earr) > 0)
                    {
                      int dsz;
                      int i;
                      char *evntmsg;

                      dsz = (d_array_get_size (earr) * LOGBUF) + 10;
                      evntmsg = (char *) calloc (dsz, sizeof (char));
                      if (evntmsg == NULL)
                        {
                          logwriter("CRITICAL: allocating evntmsg failed, calloc()");
                          return 1;
                        }

                      sprintf (evntmsg, "222 ");

                      for (i = 0; i < d_array_get_size (earr); i++)
                        strncat (evntmsg, (char *) earr->element[i]->payload, dsz);

                      strncat (evntmsg, "\r\n", dsz);
                      if (write_sock (ssl, evntmsg) > 0)
                        {
                          free (evntmsg);
                          d_array_delete (earr);
                          return -1;
                        }
                      free (evntmsg);
                    }
                  else
                    {
                      /* No active events */
                      if (write_sock (ssl, "222\r\n") > 0)
                        {
                          d_array_delete (earr);
                          return -1;
                        }
                    }
                }

              d_array_delete (earr);
              task = CONCHECK;
            }
          else if (task_n == SRVRMODPHASE)
            {
              char *payload;
              char *rulenr;
              char *operation;
              char *triggers;
              int slen;
              int i;

              payload = (char *) calloc (strlen (rxbuf) + 1, sizeof (char));
              if (payload == NULL)
                {
                   logwriter("CRITICAL: allocating event data failed, calloc()");
                   return 1;
                }

              get_message_payload (rxbuf, payload, strlen (rxbuf));
              free (rxbuf);
         
              rulenr = &payload[0];
              operation = NULL;
              triggers = NULL;

              i = 0; 
              slen = strlen (payload);

              /* Adjust pointers to event's phase and triggers */
              while (i < slen)
                {
                  if ((int) isspace (payload[i]))
                    {
                      payload[i] = '\0';
                      if (i + 1 < slen)
                        {
                          if ((int) isgraph (payload[i + 1]))
                            {
                              if (operation == NULL)
                                operation = &payload[i + 1];
                              else
                                {
                                  triggers = &payload[i + 1];
                                  break;
                                }
			    }
                        }
                    }
                  i++;
                }

              if (operation == NULL || triggers == NULL)
                {
                  if (write_sock (ssl, "509 REQUEST FAILED\r\n"))
                    {
                      free (payload);
                      return -1;
                    }
                }
              else
                {
                  char *phase;
                  int rval;

                  phase = strstr (operation, ":");
                  if (phase == NULL)
                    {
                      if (write_sock (ssl, "509 REQUEST FAILED\r\n") > 0)
                        {
                          free (payload);
                          return -1;
                        }
                    }
                  else
                    {
                      /* Adjust at event's subsequent phase */
                      phase++;

                      /* "R" means that next phase will be runned */
                      if (!strncmp (operation, "R", 1))
                        {
                          rval = modify_event_phase (vars, atoi (rulenr), atoi (phase), triggers, RUNPHASE); 
                          if (write_sock (ssl, rval == 0 ? "232 OK\r\n" :
                                          "509 REQUEST FAILED\r\n") > 0)
                            {
                              free (payload);
                              return -1;
                            }
                        }
                      /* "L" means that event will be locked at it's current phase */
                      else if (!strncmp (operation, "L", 1))
                        {
                          rval = modify_event_phase (vars, atoi (rulenr), atoi (phase), triggers, LOCKPHASE);
                          if (write_sock (ssl, rval == 0 ? "232 OK\r\n" :
                                          "509 REQUEST FAILED\r\n") > 0)
                            {
                              free (payload);
                              return -1;
                            }
                        }
                      /* "U" means that event will be unlocked  */
                      else if (!strncmp (operation, "U", 1))
                        {
                          rval = modify_event_phase (vars, atoi (rulenr), atoi (phase), triggers, ULCKPHASE);
                          if (write_sock (ssl, rval == 0 ? "232 OK\r\n" :
                                          "509 REQUEST FAILED\r\n") > 0)
                            {
                              free (payload);
                              return -1;
                            }
                        }
                      /* Reject request otherwise */
                      else
                        {
                          if (write_sock (ssl, "505 REQUEST REJECTED\r\n"))
                            {
                              free (payload);
                              return -1;
                            }
                        }
                    }
                 }
              free (payload);              
              task = CONCHECK;
            }
	  else if (task_n == UNKNOWN)
	    {
	      /* Received something weird, send error message and
	       * switch to STARTCOM state
	       */
	      if (write_sock (ssl, "507 UNKNOWN MESSAGE\r\n") > 0)
		return -1;
	      task = STARTCOM;
	    }
          else if (task_n == REQRJCT)
            {
              /* Reject node manager request */
              if (write_sock (ssl, "505 REQUEST REJECTED\r\n") > 0)
                return -1;
              task = CONCHECK;
            }
	}
    }
}

/* Extracts message payload by removing protocol opcode and delimiter */
static void
get_message_payload (char *srcbuf, char *dstbuf, unsigned int dstlen)
{
  char *cptr;
  char *fptr;
  char **lptr;
  unsigned int i;

  fptr = &dstbuf[0];

  /* calculate length */
  lptr = &srcbuf + strlen(srcbuf) ;
  /* adjust fptr to message payload */
  cptr = &srcbuf[0];
  while (!isspace ((int) *cptr))
    cptr++;

  cptr++;
  
  i = 0;
  while (1)
    {
      //if (!isgraph ((int) *cptr) || !isspace ((int) *cptr))
      //  break;

      if (*cptr == '\r' && *(cptr + 1) == '\n')
        {
         /*if ((long)&cptr < (long)&lptr){
          break;
	  } */
        }
      if (i >= dstlen)
        break;
      else
        *fptr++ = *cptr++;

      i++;
    }
}

static int
save_sensor_file (char *fname, char *fcontents)
{
  FILE *fp;
  char *buptr;

  buptr = strstr (fname, ".backup"); 
  if (buptr == NULL)
    {
      /* Take backup from working configuration file */

      char *bufile;

      bufile = (char *) calloc (strlen (fname) + 10, sizeof (char));
      if (bufile == NULL)
        {
          logwriter("CRITICAL: save_sensor_file failed, calloc()");
          return -1;
        }

      sprintf (bufile, "%s.backup", fname);
      rename (fname, bufile);
      
      free (bufile);
    }
  else
    {
      /* We do not want to back up broken configuration
       * file because we are already running with working
       * .backup configuration file
       */ 
      *buptr = '\0';
    }

  fp = fopen (fname, "w");
  if (fp == NULL)
    return 1;

  if (write (fileno (fp), fcontents, strlen (fcontents)) < 0)
    return 1;
  if (fsync (fileno (fp)) < 0)
    return 1;

  fclose (fp);

  return 0;
}


static int
handle_operation (SSL * ssl, char *param, enum msgtype type, char *payload,char *whitelistfile, char *offset_container_file)
{
  int wval;
  char *fmsg;
  
  if (type == SRVRSENDCMD){
    fmsg = exec_sensor_cmd_op (type, param, payload,whitelistfile);
  }else{
    fmsg = exec_sensor_file_op (param, type, payload, offset_container_file);
  }

  if (fmsg != NULL)
    {
      /* Message was constructed by exec_sensor_file_pipe_op, send
       * it now to node manager
       */

      wval = write_sock (ssl, fmsg);
      free (fmsg);
      if (wval != 0){
	return -1;
     }
    }
  else
    {
      /* Ouch ... */
      if (write_sock (ssl, "509 REQUEST FAILED\r\n") > 0)
	return -1;
    }
  return 0;
}

static char*
get_prog_name_from_path (char *progpath){

  char myprogpath[BUFSIZE] = {0};
  strcpy(myprogpath,progpath);
  char *progname;
  progname = (char*) calloc (BUFSIZE, sizeof (char));
  if (progname == NULL)
    {
     logwriter("CRITICAL: exec_sensor_file_op failed, calloc()");
      return NULL;
   }

  char *token              = strtok(myprogpath, "/");
  while (token != NULL){
     snprintf(progname, sizeof(token)+1, "%s", token);
     token = strtok(NULL, "/");
  }
  return progname;
}


static int
is_cmd_whitelisted(char *cstring,char *whitelistfile){
   FILE *fpc;
   char mycmd[BUFSIZE] = {0};
   int exec_action = 0;

   fpc  = fopen (whitelistfile, "r");
   if (fpc == NULL)
     return -1;

   while (fgets (mycmd, sizeof (mycmd), fpc) != NULL ){
     mycmd[strlen(mycmd) - 1] = 0;
     /* skip empty rows and rows starting with # */
     if (mycmd[0] != '\0' && !isspace(mycmd[0]) && mycmd[0] != '#' ){
       /* allow any command to be run  */
       if (mycmd[0] == '*'){
         exec_action = 1;
         break;
       }

       /* disallow combining commands with  */
       if (strstr(cstring,"&" ) == NULL && strstr(cstring,";" ) == NULL ){
        /* check if command is whitelisted */
         if (strncmp(mycmd, cstring, strlen(mycmd)) == 0){
           exec_action = 1;
           break;
         }
       }
     }
   }

   fclose (fpc);
   return exec_action;
}

static char *
exec_sensor_cmd_op (enum msgtype type, char* ruledir, char *payload,char *whitelistfile)
{
   char *job_out;
   char *paramsArr[]   = {NULL, NULL, NULL, NULL,NULL};
   const int MAXPARAMS = 6;
   int param_indx      = 1;

   char cmd_outfile[STRBUF] = {0};
   char cmd_job_out[STRBUF] = {0};
   int  redirect_stdout     = 1; 
   char *mypayload;
   mypayload = (char*) calloc (BUFSIZE, sizeof (char));
   strcpy(mypayload,payload);
  
   char *token              = strtok(payload, " ");
   char *cmd_to_run         = token;

   char *mycmd_cr = strstr(cmd_to_run, "\n");
   if (mycmd_cr != NULL){
      cmd_to_run[ strlen(cmd_to_run) - 1 ] = '\0';
   }

   char *mycmd_ret = strstr(cmd_to_run, "\r");
   if (mycmd_ret != NULL){
      cmd_to_run[ strlen(cmd_to_run) - 1 ] = '\0';
   }

   char *progname  =  get_prog_name_from_path (cmd_to_run);
   paramsArr[0]    = progname;

   token              = strtok(mypayload, " ");
   token = strtok(NULL, " ");
   while (token != NULL){
     if ( strstr(token, "\n") != NULL)
        token[strlen(token) - 1] = 0;
     if ( strstr(token, "\r") != NULL)
        token[strlen(token) - 1] = 0;

     if (strcmp(token,"--nuhed_outfile") == 0){
        token = strtok(NULL, " ");
        if (token != NULL && strstr(token,";") == NULL && strstr(token,"&") == NULL && strstr(token,">") == NULL && strstr(token,"/") == NULL ){
           token++;
           snprintf(cmd_outfile, sizeof(cmd_outfile), "%s%s%s", ruledir, "/logs/", token);
        }
        token = strtok(NULL, " ");
        continue;
      }
		  
      if (strcmp(token,"--job_id") == 0){
          token = strtok(NULL, " ");
          if (token != NULL && strstr(token,";") == NULL && strstr(token,"&") == NULL && strstr(token,">") == NULL && strstr(token,"/") == NULL ){
	    snprintf(cmd_job_out, sizeof(cmd_job_out), "%s%s%s", ruledir, "/logs/", token);
	  }
	  token = strtok(NULL, " ");
          continue;
      }
                   
      if (strcmp(token,"--no-stdout") == 0){
	   redirect_stdout = 0;		   
	    token = strtok(NULL, " ");
           continue;
      }

      if (param_indx < MAXPARAMS){
	    paramsArr[param_indx] = token;
	    param_indx = param_indx + 1;
	    token = strtok(NULL, " ");
      }
  } 


  char *fmsg;
  fmsg = (char *) calloc (BUFSIZE, sizeof (char));
  if (fmsg == NULL)
    {
     logwriter("CRITICAL: exec_sensor_file_op failed, calloc()");
      return NULL;
    }

  /* command is whitelisted ? */
  if (is_cmd_whitelisted(cmd_to_run,whitelistfile) != 1){
    snprintf (fmsg, BUFSIZE, "510 ");
    return fmsg;
  }
  
  /* file exists */
  if (access(cmd_to_run, F_OK) == 0) {
    // file exists
  } else {
    snprintf (fmsg, BUFSIZE, "511 ");
    return fmsg;
  }

  int pid = fork();
  if ( pid == 0 ) {
     // open stdout to custom log file
     if (strcmp(cmd_outfile,"") != 0 && redirect_stdout == 1){
       fprintf(stderr," Running custom stdout for %s\n",cmd_to_run);
       int fd = open(cmd_outfile, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
       dup2(fd, 1);
       dup2(fd, 2);
       close(fd);
     }else{
       fprintf(stderr," RUnning default stdout for %s\n",cmd_to_run);
        // by default stdout / stderr NUHEDDIR/logs/{job_id}_{exec_time}
       int fd = open(job_out, O_RDWR | O_CREAT, S_IRUSR | S_IWUSR);
       dup2(fd, 1);
       dup2(fd, 2);
       close(fd);
     }

     /*for(size_t i = 0; i < sizeof(paramsArr) ; i++)
     {
	     fprintf(stderr,"Param: %i Value:%s \n",i,paramsArr[i]);
     }
     char *args[] = {"rkhunter","-c", "-q", NULL,NULL,NULL};
*/
     char data[STRBUF] = { 0 };

      snprintf(data, sizeof (data), "Info: Executing Job cmd %s %s", cmd_to_run,payload);
      logwriter (data);

      execv(cmd_to_run, paramsArr);

      fprintf(stderr," Execl for  %s payload %s FAILED\n",cmd_to_run,payload);
      snprintf(data, sizeof (data), "Errror: Executing Job cmd %s %s FAILED", cmd_to_run,mypayload);
      logwriter (data);
  } 
  snprintf (fmsg, BUFSIZE, "178 %i",pid);
  free(mypayload);
  free(progname);
  return fmsg;
}
 
static long 
read_logfile_offset (char *offset_container_file)
{
   FILE *fp;
   char buffer[512] = { 0 };

   fp  = fopen (offset_container_file, "r");
   if (fp != NULL){
     fgets (buffer, sizeof (buffer), fp);
     fclose(fp);
   }
   return atol(buffer);
}

static void
write_logfile_offset (char *offset_container_file, char* offset)
{
   FILE *fpo;
   fpo  = fopen (offset_container_file, "w");
   if (fpo != NULL){
     fprintf (fpo,offset);
     fclose(fpo);
   }
}

static char *
exec_sensor_file_op (char *param, enum msgtype type,char *payload,char *offset_container_file)
{
  FILE *fp;
  char buffer[512] = {
    0
  };
  char *newp;
  unsigned int r;

  long log_offset;
  long log_end_offset;
  char cpy[11];
 
  char *fmsg;
  fmsg = (char*) calloc (BUFSIZE, sizeof (char));
  if (fmsg == NULL)
    {
     logwriter("CRITICAL: exec_sensor_file_op failed, calloc()");
      return NULL;
   }
 

  /* Request external command (pipe) or regular file */
  fp  = fopen (param, "r");
  if (fp == NULL)
    return NULL;

  if (offset_container_file != NULL)
   {
      log_offset = read_logfile_offset (offset_container_file);

      // If nuhed.log re-created start from position 0
      fseek(fp,0L,SEEK_END);
      log_end_offset = ftell(fp);
      if (log_offset > log_end_offset)
        fseek(fp,0,SEEK_SET);
      else
        fseek(fp,log_offset,SEEK_SET);
   }

   /* Prefix message with protocol opcode */
   if (type == SRVRGETRN)
     snprintf (fmsg, BUFSIZE, "164 ");
   else if (type == SRVRREQCN)
     snprintf (fmsg, BUFSIZE, "172 ");
   else if (type == SRVRREQLOG)
     snprintf (fmsg, BUFSIZE, "176 ");
   else if (type == SRVRREQACTRULES)
     snprintf (fmsg, BUFSIZE, "180 ");

  r = 1;
  while (fgets (buffer, sizeof (buffer), fp) != NULL ){
    newp = (char *) calloc  (r * BUFSIZE, sizeof (char));
    if (newp == NULL){
      logwriter("CRITICAL: exec_sensor_file_op failed, calloc()");
      return NULL;
    }

    strcpy (newp, fmsg);
    free (fmsg);
    fmsg = newp;

    // skip '[' with + 1
    memcpy(cpy,buffer + 1,11);

    // return rows by date 
    if (payload != NULL && 1==2){ 
      if ( !strncmp (cpy, payload, 10)){
        strncat (fmsg, buffer, BUFSIZE);
      }
    } else {
       if (type == SRVRREQLOG){

          if (strstr (buffer, "[ACTION]") != NULL || strstr (buffer, "[ALERT]") != NULL ){
              strncat (fmsg, buffer, BUFSIZE);
	      r++;
	  }
          if (r > 80 || offset_container_file != NULL && ftell(fp) > log_end_offset)
            break;
      }else{
         strncat (fmsg, buffer, BUFSIZE);
	 r++;
      }
    }
  }


  if (offset_container_file != NULL){
    sprintf(buffer, "%ld", ftell(fp));
    write_logfile_offset (offset_container_file,buffer);
  }

  fclose (fp);
   
  /* Terminate message with CRLF delimiter */
  strncat (fmsg, CRLF, r * BUFSIZE);
  return fmsg;
}

/* Returns correct task type for message opcode */
static enum msgtype
get_message_opcode (char *msg, struct internal *vars)
{
  /* Do not process stubb messages */
  if (strlen (msg) < 3)
    return UNKNOWN;

  /* If we are in FREEZE state only unfreezing is allowed */
  if (vars->run_stat == FREEZE)
    {
      if (!strncmp (msg, "213", 3))
        /* We're free to go! */
        return SRVRREQUNFRZ;

      return REQRJCT;
    }

  fprintf(stderr, " ---------- GOT MSG: %s",msg);
  /* General messages for RUNNING and DEFUNC state
   ***********************************************
   */ 
  if (!strncmp (msg, "152", 3))
    /* Node manager has replied to our hello message */
    return SRVRHELLO;
  else if (!strncmp (msg, "153", 3))
    /* Node manager sends heartbeat for us */
    return SRVRPING;
  else if (!strncmp (msg, "161", 3))
    /* Node manager has requested our rulesets */
    return SRVRREQRS;
  else if (!strncmp (msg, "163", 3))
    /* Send specific ruleset for node manager */
    return SRVRGETRN;
  else if (!strncmp (msg, "165", 3))
    /* Node manager uploads specific ruleset for us */
    return SRVRSENDRN;
  else if (!strncmp (msg, "171", 3))
    /* Node manager requested our configuration file */
    return SRVRREQCN;
  else if (!strncmp (msg, "173", 3))
    /* Node manager uploads configuration file for us */
    return SRVRSENDCN;
  else if (!strncmp (msg, "175", 3))
    /* Node manager returns log file  */
    return SRVRREQLOG;
  else if (!strncmp (msg, "179", 3))
    /* Node manager returns active rules  */
    return SRVRREQACTRULES;
  else if (!strncmp (msg, "177", 3))
    /* Node manager   runs cmd*/
    return SRVRSENDCMD;
  else if (!strncmp (msg, "193", 3))
    /* Node manager returns PID status of running cmd  */
    return SRVRREQPIDSTATUS;
  else if (!strncmp (msg, "201", 3))
    /* Node manager request restart */
    return SRVRREQBOOT;
  else if (!strncmp (msg, "401", 3))
    /* Node manager returns log file  */
    return SRVRSENDLOG;
  else if (!strncmp (msg, "393", 3))
    /* Node manager disconnects us */
    return SRVRREQQUIT;

  /* DEFUNC state does not support all messages */
  if (vars->run_stat == DEFUNC)
    return REQRJCT;
  else
    {
      if (!strncmp (msg, "181", 3))
        /* Node manager has sent alert request */
        return SRVRREQALRT;
      else if (!strncmp (msg, "211", 3))
        /* Node manager freezes us */
        return SRVRREQFRZ;
      else if (!strncmp (msg, "221", 3))
	/* Node manager request active events */
	return SRVRREQEVNTS;
      else if (!strncmp (msg, "231", 3))
        /* Node manager request to run next phase for
         * event or lock/unlock event phase
         */
        return SRVRMODPHASE;

      return UNKNOWN;
    }
}

/* Check if we have pending alert messages and send them 
 * to node manager
 */
static int
send_pending_messages (SSL * ssl)
{
  struct node *current;
  struct stat buf;
  char *alarmstr;
  char *wiremsg;

  pthread_mutex_lock (&msglist_m);
  /* Check if we have messages in failsafe store and send them first */
  if ((stat (failstore, &buf) != -1))
    {
      FILE *fp;
      char buffer[LOGBUF];

      fp = fopen (failstore, "r");
      if (fp == NULL)
        return 1;

      while (fgets (buffer, sizeof (buffer), fp) != NULL)
        {
          wiremsg = (char *) calloc (strlen (buffer) + 10, sizeof (char));
          if (wiremsg == NULL)
            {
              logwriter("CRITICAL: allocating alarm message for node manager failed, calloc()");
              pthread_mutex_unlock (&msglist_m);
              return 1;
            }

          snprintf (wiremsg, strlen (buffer) + 10, "191 %s\r\n", buffer);
          if (write_sock (ssl, wiremsg) > 0)
            {
              free (wiremsg);
              pthread_mutex_unlock (&msglist_m);
              return -1;
            }
          free (wiremsg);
        }
      fclose (fp);

      /* All alerts sent and we're connected to node manager; get rid of
       * failsafe file
       */
      unlink (failstore);
    }

  /* Process alerts from msglist */
  current = msglist->head;
  while (current != NULL)
    {
      alarmstr = (char *) current->payload;
      wiremsg = (char *) calloc (strlen (alarmstr) + 10, sizeof (char));
      if (wiremsg == NULL)
        {
          logwriter("CRITICAL: allocating alarm message for node manager failed, calloc()");
          pthread_mutex_unlock (&msglist_m);
          return 1;
        }

      snprintf (wiremsg, strlen (alarmstr) + 10, "191 %s\r\n", alarmstr);
      if (write_sock (ssl, wiremsg) > 0)
	{
	  free (wiremsg);
	  pthread_mutex_unlock (&msglist_m);
	  return -1;
	}
      free (wiremsg);
      current = current->next;
    }

  /* Reset msglist */
  slist_clear (msglist);
  pthread_mutex_unlock (&msglist_m);

  return 0;
}

/* Add alerts received from Nuhe sensor to msglist and
 * process them later in CONCHECK state
 */
int
send_message_to_nodemgr (const char *msg)
{
  struct node *new_node;

  pthread_mutex_lock (&msglist_m);
  /* We're not connected to node manager; save alerts to failsafe store */
  if (connected < 1)
    {
      if (alert_failsafe_write (msg) > 0)
        {
          pthread_mutex_unlock (&msglist_m);
	  logwriter ("Error: failed to store alerts to failsafe store");
          return 1;
        } 
    }
  else
    {
      new_node = slist_append (msglist);
      if (new_node == NULL)
        {
          pthread_mutex_unlock (&msglist_m);
          return -1;
        }

      new_node->payload = (char *) calloc (strlen (msg) + 1, sizeof (char));
      if (new_node->payload == NULL)
        {
          pthread_mutex_unlock (&msglist_m);
          return -1;
        }

      memcpy (new_node->payload, msg, strlen (msg));
    }

  pthread_mutex_unlock (&msglist_m);
  return 0;
}

/* Write alerts destined to node manager to failsafe store, when
 * connection to node manager is broken
 */
static int
alert_failsafe_write (const char *msg)
{
  FILE *fp;

  fp = fopen (failstore, "a");
  if (fp == NULL)
    return 1;

  fprintf (fp, "%s\n", msg);
  fclose (fp);

  return 0;
}

/* Move all alerts currently at msglist to failsafe store */
static int
move_alerts_to_failsafe ()
{
  struct node *current;
  int rval;

  rval = 1;

  if (msglist == NULL)
    return rval;

  pthread_mutex_lock (&msglist_m);
  current = msglist->head;
  if (current != NULL)
    {
      rval = 0;
      while (current != NULL)
        {
          if (alert_failsafe_write ((char *) current->payload) > 0)
            logwriter ("Error: failed to store alerts to failsafe store");
          current = current->next;
        }

      /* Reset msglist */
      slist_clear (msglist);
    }
  pthread_mutex_unlock (&msglist_m);

  return rval;
}
