# ThreadPool.py v. 0.02
#
# Copyright (C) 2007 Tuomo Makinen (tjam@users.sourceforge.net)
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

import threading
#from multiprocessing import Queue
import queue
import sys
import time
import copy
import os

class ThreadPool:
    def __init__ (self, minthreads=5, startthreads=5, maxthreads=20):
       if (minthreads < 1):
	       raise (ThreadPoolError, "Minimum number of threads is 1")
       elif (maxthreads < minthreads):
	       raise (ThreadPoolError, "Minimum is greater than maximum")
       else:
	       # Number of threads in pool
         self.__minthreads = minthreads
         self.__startthreads = startthreads
         self.__maxthreads = maxthreads
	       # Flag to indicate if pool is started
         self.__started = False
         # Initialize JobQueue for worker threads
         self.__job_queue =  queue.Queue(self.__maxthreads)
         # Condition variable to indicate queue state
         self.__queue_get_job = threading.Condition()
	       # Destroy "flag" for thread pool
         self.__destroy_threadpool = False
	       # list for workerthread
         self.__workerthreads = []
         # list containing active threads
         self.__activethreads = []
	       # list containing currently idle threads
         self.__idlethreads = []
	       # Keepalive time for idle threads (value in milliseconds)
         self.__keepalive = 5000
	       # Try to start as many threads, if all threads are occupied
         self.__numtostart = 5
	       # Flag to indicate if idle threads should be destroyed
         self.__cleanpool = False
         self.__tc = 0

    def start (self):
      if self.__started:
        raise (ThreadPoolError, "ThreadPool is already started")
      else:
	     # Start worker threads
         for thread in range(self.__startthreads):
	          self.__addThread()
         # Start monitoring thread for thread pool
         self.__monitorthread = threading.Thread(group=None, target=self.__monitorPool, args=())

         self.__monitorthread.start()
         self.__started = True

    def runTask (self, callback, *args, **kwargs):
      if not self.__started:
            raise (ThreadPoolError, "ThreadPool is not started")

      while True:
            self.__queue_get_job.acquire()
            psize = self.getPoolSize()
            if (self.numActiveThreads() >= psize and \
               psize < self.__maxthreads):
                # Server is overloaded -> add new threads to pool
                # start new threads for pool if poolsize < maxthreads
                for i in range(self.__numtostart):
                    if (len(self.__workerthreads) < self.__maxthreads):
                        self.__addThread()

                self.__queue_get_job.notify()
                self.__queue_get_job.release()
            else:
                work_unit = (callback, args, kwargs)
                #self.__job_queue.put_nowait(work_unit)
                self.__job_queue.put(work_unit)
                self.__queue_get_job.notify()
                self.__queue_get_job.release()
                break

    def __addThread (self):
        t = threading.Thread(group=None, target=self.__checkJobs, args=())
        self.__workerthreads.append(t)
        t.start()

    def __monitorPool (self):
      """ This function runs pool monitoring thread.
      Monitoring thread checks idle threads of the pool 
      and signals worker threads if pool cleaning
      is needed.
      """
      while True:
        if self.__destroy_threadpool:
          break

        time.sleep ((self.__keepalive/1000))
        self.__queue_get_job.acquire()
        if (len(self.__workerthreads) > self.__minthreads and len(self.__activethreads) < self.__minthreads):
          for thread in self.__workerthreads:
             if thread not in self.__activethreads:
                self.__idlethreads.append(thread)

                if ((len(self.__workerthreads) - len(self.__idlethreads)) <= self.__minthreads):
                  break

          idlethreads = copy.copy(self.__idlethreads)
          self.__cleanpool = True
          self.__queue_get_job.notifyAll()
          self.__queue_get_job.release()
		
          for i in range(len(idlethreads)):
            idlethreads[i].join()
        else:
          self.__queue_get_job.release()
		
    def __checkJobs (self):
      me = threading.currentThread()
      while True:
            self.__queue_get_job.acquire()
            while (self.__destroy_threadpool == False and self.__job_queue.empty() == True):
              self.__queue_get_job.wait()

              if self.__cleanpool:
                break
              if self.__destroy_threadpool:
                break
		
            if self.__cleanpool:
                if me in self.__idlethreads:
                  self.__workerthreads.remove(me)
                  self.__idlethreads.remove(me)
                  if (len(self.__idlethreads) == False):
                    self.__cleanpool = False
                    
                  self.__queue_get_job.release()
                  sys.exit()
                else:
                  self.__queue_get_job.release()
                  continue
  
            if self.__destroy_threadpool:
                self.__queue_get_job.release()
                sys.exit()
            else:
                work_unit = self.__job_queue.get_nowait()
                self.__queue_get_job.release()
		
                # Add thread to list of active threads
                self.__activethreads.append(me)
                # Handle threads work
                work_unit[0] (*work_unit[1], **work_unit[2])
                # Remove thread from list of active threads
                self.__activethreads.remove(me)
                self.__tc = self.__tc + 1

    def stop (self):
      """ Shutdown threadpool by setting __destroy_threadpool to True
      """
      if not self.__started:
         raise (ThreadPoolError, "ThreadPool is not started")
      else:
          if not self.__destroy_threadpool:
                self.__queue_get_job.acquire()
                self.__destroy_threadpool = True
                self.__queue_get_job.notifyAll()
                self.__queue_get_job.release()
                # Join our workers
                for thread in self.__workerthreads:
                    thread.join()

                # Join monitoring thread
                self.__monitorthread.join()
                self.__workerthreads = []
                self.__destroy_threadpool = False
                self.__started = False

    def Lock (self):
        return threading.Lock()
 
    def RLock (self):
        return threading.RLock()

    def acquireLock (self, lock):
        lock.acquire()

    def releaseLock (self, lock):
        lock.release()

    def isStarted (self):
   	    return self.__started

    def getThreadId (self):
        return threading.currentThread()

    def getMaxThreads (self):
	      return self.__maxthreads

    def getMinThreads (self):
	      return self.__minthreads

    def getPoolSize (self):
	      return len(self.__workerthreads)
   
    def getProcessedTasks (self):
        return self.__tc

    def setKeepAliveTime (self, mseconds):
    	self.__keepalive = mseconds
 
    def getKeepAliveTime (self):
    	return self.__keepalive

    def numActiveThreads (self):
        return len(self.__activethreads)

    def numIdleThreads (self):
        return len(self.__idlethreads)

class ThreadPoolError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)

class QueueFullError(Exception):
    def __init__(self, value):
        self.value = value
    def __str__(self):
        return repr(self.value)
